unit GraphUtils;

interface

uses Contnrs, dialogs, sysutils, Classes, math, StrUtils;

type

  TNode = class
    private
        fTerminal: Boolean;
        fLength: Integer;
        fSuffixLink: TNode;
        fEdges: Array[Char] of TNode;
        fQueueNext: TNode;
        fGS1: Integer;
        fGS2: Integer;
    public
        Idx: Integer;
        OutputList: TList;

        property SuffixLink: TNode read fSuffixLink write fSuffixLink;
        property Terminal: Boolean read fTerminal write fterminal;
        property Len: Integer read fLength write fLength;
        property GS: Integer read fGS1;

        constructor Create;//(EdgeCount: Integer);
        destructor Destroy; Override;
        
        // Kante ermitteln
        function GetTarget(c: Char): TNode;
        // Kante setzen
        procedure SetTarget(c: Char; Dest: TNode);
        // Knoten Kopieren
        procedure Assign(aNode: TNode);
  end;

  TNodeQueue = class
    private
      flast: TNode;
    public
      First: TNode;
      procedure Insert(new: TNode);
      function Delete: TNode;
  end;

  TDawg = class
      private
        fNodes: TObjectlist; // zum spteren freigeben der Nodes
        fInit: TNode;
        fLast: TNode;
        fcount: Integer; // Zum Zhlen der eingefgten Knoten
      public
        property Init: TNode read fInit;
        property Last: TNode Read fLast write fLast;

        constructor Create;
        destructor Destroy; Override;

        // Lscht den DAWG bis auf den Init-Knoten
        procedure Clear;
        // Baut den DAWG fr einen String auf.
        Procedure Build(p: String);
        function Output: String;
  end;

  TDawgRecord = record
      Init: Integer;  // Der Startknoten
     // NodeCount: Integer; // Zhlt die eingefgten Knoten
      Len: Array of Integer;        // Speichert die Lnge der Knoten
      Terminal: Array of Boolean;   // ... die Eigenschaft Terminal  der Knoten
      SuffixLink: Array of Integer; // ... die Suffixlinks der Knoten
      Target:  Array of Array[Char] of Integer;  // Kanten, Gre: |Nodes| * 256
  end;

  TTrie = class
    private
      fNodes: TObjectlist; // zum spteren freigeben der Nodes
      fRoot: TNode;
      fcount: Integer; // Zum Zhlen der eingefgten Knoten
    public
      property Root: TNode read fRoot;

      constructor Create;
      destructor Destroy; Override;

      // Lscht den Trie bis auf den Init-Knoten
      procedure Clear;
      // Baut den Trie fr eine Liste von String auf.
      Procedure Build(p: TStrings);
      function Output: String; virtual;
  end;

  TAhoTrie = Class(TTrie)
    public
      procedure BuildAho(p: TStrings);
      procedure BuildAdvancedAho(p: TStrings);
      function Output: String; override;
  end;

  TCommentzWalterTrie = Class(TTrie)
    private
      flmin: Integer;
    public
      property lmin: Integer read flmin;
      procedure BuildCW(p: TStrings);
      function Output: String; override;
  end;

  TMultiOracle = Class(TTrie)
    public
      procedure BuildOracle(p: TStrings);
      function Output: String; override;
  end;

  function BuildDawg(p:String): TDawgRecord;
  function OutputDawg(aDawg: TDawgRecord): String;


implementation

constructor TNode.Create;//(EdgeCount: Integer);
begin
  inherited create;
  fTerminal := False;
  fLength := 0;
  Idx := 0;
  fSuffixLink := Nil;
end;

destructor TNode.Destroy;
begin
  if assigned(OutputList) then
    OutputList.Free;
end;

// Kante ermitteln
function TNode.GetTarget(c: Char): TNode;
begin
  result := fEdges[c]
end;

// Kante setzen
procedure TNode.SetTarget(c: Char; Dest: TNode);
begin
  fEdges[c] := Dest
end;

// Knoten Kopieren
procedure TNode.Assign(aNode: TNode);
var c: Char;
begin
  for c := Low(Char) to High(Char) do
    fEdges[c] := aNode.fEdges[c];

  fTerminal := aNode.Terminal;
  fLength := aNode.Len;
  fSuffixLink := aNode.SuffixLink;
end;

// ------------------------------------------------
// Klasse TNodeQueue
// ------------------------------------------------

procedure TNodeQueue.Insert(new: TNode);
begin
  if assigned(fLast) then
    fLast.fQueueNext := new
  else
    First := new;
  fLast := new;
  new.fQueueNext := Nil;
end;

function TNodeQueue.Delete: TNode;
begin
  result := First;
  if result = fLast then fLast := Nil;
  if assigned(First) then
    First := First.fQueueNext;
end;

// ------------------------------------------------
// Klasse TDawg
// ------------------------------------------------

constructor TDawg.Create;
begin
  fInit := TNode.Create;
  fLast := fInit;

  // Liste erzeugen und Kapazitt auf NodeCount setzen - ist quasi ein Array,
  // Kann aber automatisch erweitert werden ;-)
  fNodes := TObjectlist.Create(True);
end;

destructor TDawg.Destroy;
begin
  fNodes.Free;
  fInit.Free;
end;

procedure TDawg.Clear;
begin
  fNodes.Clear;
  fInit.Free;
  fInit := TNode.Create;
  fLast := fInit;
  fCount := 0;
end;

Procedure TDawg.Build(p: String);
var m, i: Integer;
    u, v, w, r: TNode;
begin
  m := Length(p);
  // ggf. alten DAWG vorher lschen
  Clear;
  fNodes.Capacity := 2*m +2;

  for i := 1 to m do
  begin
      u := fLast;
      // Neuen Knoten erzeugen
      v := TNode.Create;
      inc(fCount);
      v.Idx := fCount;
      // neuen Knoten in Liste einfgen, zum spteren Aufrumen
      fNodes.Add(v);
      v.Len := u.Len + 1;

      While (u <> fInit) and (u.GetTarget(p[i]) = NIL) do
      begin
          u.SetTarget(p[i],v);
          u := u.SuffixLink;
      end;

      if u.GetTarget(p[i]) = NIl then
      begin
          // wir sind am Startknoten angekommen
          fInit.SetTarget(p[i],v);
          v.SuffixLink := fInit;
      end else
      begin
          // So eine Kante existiert
          w := u.GetTarget(p[i]);
          if u.Len + 1 = w.Len then
              // (u,w) ist feste Kante
              v.SuffixLink := w
          else
          begin
              // (u,w) ist lose Kante
              r := TNode.Create;
              inc(fCount);
              r.Idx := fCount;
              fNodes.Add(r);
              r.Assign(w);
              r.Len := u.Len + 1;
              w.SuffixLink := r;
              v.SuffixLink := r;
              while (u <> NIL) and (u.GetTarget(p[i]) = w{<> NIL}) do
              begin
                  u.SetTarget(p[i],r);
                  u := u.SuffixLink;
              end;
          end;
      end;
      fLast := v;
  end;// for i =1 to m

  flast.Terminal := True;
  While fLast <> fInit do
  begin
      fLast := fLast.SuffixLink;
      fLast.Terminal := True;
  end;
end;

function TDawg.Output: String;
var v: Integer;
    e: Char;
begin
  result := '';
  // fInit ausgeben
  for e := Low(Char) to High(Char) do
    begin
        if fInit.fEdges[e] <> Nil then
          result := result + Format('Kante von %d nach %d mit Markierung %s', [
          fInit.Idx,
          fInit.GetTarget(e).Idx,e]) + #13#10
    end;
  result := result + Format('Ende Knoten %d, (%s)-------------',
                     [fInit.Idx, BoolToStr(fInit.Terminal, True)])
                   + #13#10;

  for v := 0 to fnodes.Count - 1 do
  begin
    for e := Low(Char) to High(Char) do
    begin
        if TNode(fNodes[v]).fEdges[e] <> Nil then
          result := result +  Format('Kante von %d nach %d mit Markierung %s', [
          TNode(fNodes[v]).Idx,
          TNode(fNodes[v]).GetTarget(e).Idx, e]) + #13#10
    end;
    result := result + Format('Ende Knoten %d, (%s)-------------',
                      [TNode(fNodes[v]).Idx, BoolToStr(TNode(fNodes[v]).Terminal, True)])
                     + #13#10;
  end;
end;


function BuildDawg(p:String): TDawgRecord;
var m,i,NodeCount: Integer;
  last, u, v, w, r: Integer;
  c:Char;
  CopyC: Char;
begin
  m := length(p);
  Setlength(result.Len, 2*m+2);
  Setlength(result.Terminal, 2*m+2);
  Setlength(result.SuffixLink, 2*m+2);
  Setlength(result.Target, 2*m+2);
  result.Init := 0;
  result.SuffixLink[0] := -1;
  for c := Low(Char) to High(Char) do result.Target[0,c] := -1;
  NodeCount := 0;
  last := 0;

  for i := 1 to m do
  begin
      u := last;
      inc(NodeCount); // neuen Knoten einfgen
      v := NodeCount;
      result.Len[v] := result.Len[u] + 1;
      // Knoten initialisieren
      result.Terminal[v] := False;
      for c := Low(Char) to High(Char) do result.Target[v,c] := -1;

      While (u > 0) and (result.Target[u,p[i]] = -1) do
      begin
          // Kante einfgen
          result.Target[u,p[i]] := v;
          u := result.SuffixLink[u];
      end;

      if result.Target[u,p[i]] = -1  then
      begin
          // Wir sind am Startknoten
          result.Target[0,p[i]] := v;
          result.SuffixLink[v] := 0;
      end else
      begin
          // Kante existiert
          w := result.Target[u,p[i]];
          if result.Len[u] + 1 = result.Len[w] then
          begin
              // (u,w) ist feste Kante
              result.SuffixLink[v] := w;
          end else
          begin
              inc(NodeCount);
              r := NodeCount;
              result.Terminal[r] := False;

              //  Knoten w kopieren
              for CopyC := Low(Char) to High(Char) do
                result.Target[r,CopyC] := result.Target[w,CopyC];
              result.SuffixLink[r] := result.SuffixLink[w];

              result.Len[r] := result.Len[u] + 1;
              result.SuffixLink[w] := r;
              result.SuffixLink[v] := r;

              while (u >= 0) and (result.Target[u,p[i]] = w) do
              begin
                  result.Target[u,p[i]] := r;
                  u := result.SuffixLink[u];
              end;
          end;
      end; // u <> Startknoten, d.h. Kante existiert
      last := v;
  end; // for i = 1 to m
  result.Terminal[last] := True;
  while last >= 0 do
  begin
      last := result.SuffixLink[last];
      result.Terminal[last] := True;
  end;
end;

function OutputDawg(aDawg: TDawgRecord): String;
var v: Integer;
  c: Char ;
begin
  result := '';
  for v := 0 to Length(aDawg.Len)-1 do
  begin
    for c := Low(Char) to High(Char) do
    begin
        if aDawg.Target[v,c] >= 0 then
          result := result +  Format('Kante von %d nach %d mit Markierung %s', [
          v,
          aDawg.Target[v,c],
          c]) + #13#10
    end;
    result := result + Format('--------Ende Knoten %d, (%s) ',
                      [v, BoolToStr(aDawg.Terminal[v], True)])
                     + #13#10;
  end;
end;

//---------------------------------------------------------
// Klasse TTrie
//---------------------------------------------------------


constructor TTrie.Create;
begin
  fRoot := TNode.Create;
  fRoot.Idx := 0;

  // Liste erzeugen und Kapazitt auf NodeCount setzen - ist quasi ein Array,
  // Kann aber automatisch erweitert werden ;-)
  fNodes := TObjectlist.Create(True);
end;

destructor TTrie.Destroy;
begin
  fNodes.Free;
  fRoot.Free;
end;

// Lscht den Trie bis auf den Init-Knoten
procedure TTrie.Clear;
begin
  fNodes.Clear;
  fRoot.Free;
  fRoot := TNode.Create;
  fCount := 0;
end;

// Baut den Trie fr eine Liste von String auf.
Procedure TTrie.Build(p: TStrings);
var i, j, mi: Integer;
    u,new: TNode;
begin
  for i := 0 to P.Count - 1 do
  begin
      u := fRoot;
      j := 1; // Laufindex im Muster
      mi := length(p[i]);
      While (j <= mi) and (u.GetTarget(p[i][j]) <> NIL) do
      begin
          // im vorhandenen Trie weiterlaufen
           u := u.GetTarget(p[i][j]);
           inc(j);
      end;
      While j <= mi do
      begin
          // Trie weiterbauen
          new := TNode.Create;
          u.SetTarget(p[i][j], new);
              inc(fCount);
              new.Idx := fCount;
              fNodes.Add(new);
          u := new;
          inc(j);
      end;
      // Outputliste aktualisieren
      if u.Terminal then
          // letzter Knoten ist schon terminal
          // Aktuelles Muster der Liste hinzufgen
          u.OutputList.Add(Pointer(i))
      else
      begin
          // Knoten ist noch nicht terminal.
          // Output-Liste erzeugen und Muster einfgen
          u.OutputList := TList.Create;
          u.OutputList.Add(Pointer(i));
          u.Terminal := True;
      end;
  end;
end;

function TTrie.Output: String;
var v: Integer;
    e: Char;
    aus: Integer;
begin
  result := '';
  // fInit ausgeben
  for e := Low(Char) to High(Char) do
    begin
        if (fRoot.fEdges[e] <> Nil) and (fRoot.GetTarget(e)<>froot) then
          result := result + Format('rootKante von %d nach %d mit Markierung %s', [
          fRoot.Idx,
          fRoot.GetTarget(e).Idx,e]) + #13#10
    end;
  result := result + Format('Ende Knoten %d, (%s)-------------',
                     [fRoot.Idx, BoolToStr(fRoot.Terminal, True)])
                   + #13#10;

  for v := 0 to fnodes.Count - 1 do
  begin
    for e := Low(Char) to High(Char) do
    begin
        if TNode(fNodes[v]).fEdges[e] <> Nil then
          result := result +  Format('Kante von %d nach %d mit Markierung %s', [
          TNode(fNodes[v]).Idx,
          TNode(fNodes[v]).GetTarget(e).Idx,e]) + #13#10
    end;
    result := result + Format('Ende Knoten %d, (%s)-------------',
                      [TNode(fNodes[v]).Idx, BoolToStr(TNode(fNodes[v]).Terminal, True)])
                     + #13#10;
    if TNode(fNodes[v]).Terminal then
    begin
        for aus := 0 to TNode(fNodes[v]).OutputList.Count - 1 do
          result := result + 'Ausgabe: ' + Inttostr(Integer(TNode(fNodes[v]).OutputList[aus])) + ', ';
        result := result + #13#10
    end;
  end;
end;

//---------------------------------------------------------
// Klasse TAhoTrie
//---------------------------------------------------------

procedure TAhoTrie.BuildAho(p: TStrings);
var a: char;
  queue: TNodeQueue;
  v, r,state,s: TNode;
  i: Integer;
begin
  // Trie aufbauen
  Build(p);

  queue := TNodeQueue.Create;
  // Schleifen an der Wurzel aufbauen
  // und initiale Knoten in die Queue einfgen
  root.SuffixLink := root;
  for a := Low(Char) to High(Char) do
    if root.GetTarget(a) = Nil then
        root.SetTarget(a, root)
    else
    begin
        //Kante existiert schon - in die Queue einfgen
        v := root.GetTarget(a);
        v.SuffixLink := root;
        queue.Insert(v);
    end;

  // Durchlauf
  While queue.First <> NIL do
  begin
      r := queue.delete;//TNode(queue.Pop);
      for a := Low(Char) to High(Char) do
      begin
          s := r.GetTarget(a);
          if s <> Nil then
          begin
              queue.Insert(s);
              state := r.SuffixLink;
              While state.GetTarget(a) = NIL do state := state.SuffixLink;
              v := state.GetTarget(a); // v <> NIL !
              s.SuffixLink := v;

             if v.Terminal then
              begin
                  // Output-Liste von v an s anhngen
                  if not s.Terminal then
                  begin
                      s.Terminal := True;
                      s.OutputList := TList.Create;
                  end;
                  for i := 0 to v.OutputList.Count - 1 do
                    s.OutputList.Add(v.OutputList[i]);
              end;

          end;
      end;
  end;
  queue.Free;
end;

procedure TAhoTrie.BuildAdvancedAho(p: TStrings);
var a: char;
  queue: TNodeQueue;
  r,s: TNode;
begin
  // Aho-Trie aufbauen
  BuildAho(P);
  queue := TNodeQueue.Create;

  for a := Low(Char) to High(Char) do
    if root.GetTarget(a) <> Root then
        queue.Insert(root.GetTarget(a));

  // Durchlauf
  While queue.First <> NIL do
  begin
      r := queue.delete;
      for a := Low(Char) to High(Char) do
      begin
          s := r.GetTarget(a);
          if s <> Nil then
          begin
            queue.Insert(s);
            r.SetTarget(a,s);
          end else
          begin
            r.SetTarget(a,
            (r.SuffixLink).GetTarget(a)
            );
          end;
      end;
  end;
end;


function TAhoTrie.Output: String;
var v: Integer;
    e: Char;
    aus: Integer;
begin
  result := '';
  // fInit ausgeben
  for e := Low(Char) to High(Char) do
    begin
        if (fRoot.fEdges[e] <> Nil) and (fRoot.GetTarget(e)<>froot) then
          result := result + Format('rootKante von %d nach %d mit Markierung %s', [
          fRoot.Idx,
          fRoot.GetTarget(e).Idx,e]) + #13#10
    end;
  result := result + Format('Ende Knoten %d, (%s)-------------',
                     [fRoot.Idx, BoolToStr(fRoot.Terminal, True)])
                   + #13#10;

  for v := 0 to fnodes.Count - 1 do
  begin
    for e := Low(Char) to High(Char) do
    begin
        if TNode(fNodes[v]).fEdges[e] <> Nil then
          result := result +  Format('Kante von %d nach %d mit Markierung %s', [
          TNode(fNodes[v]).Idx,
          TNode(fNodes[v]).GetTarget(e).Idx,e]) + #13#10
    end;
    result := result + Format('Ende Knoten %d, (Suffixlink nach %d)-------------',
                      [TNode(fNodes[v]).Idx, TNode(fNodes[v]).SuffixLink.Idx])
                     + #13#10;
    if TNode(fNodes[v]).Terminal then
    begin
        result := result + 'Ausgabe: ';
        for aus := 0 to TNode(fNodes[v]).OutputList.Count - 1 do
          result := result + Inttostr(Integer(TNode(fNodes[v]).OutputList[aus])) + ', ';
        result := result + #13#10;
    end;
  end;
end;


//---------------------------------------------------------
// Klasse TCommentzWalterTrie
//---------------------------------------------------------

procedure TCommentzWalterTrie.BuildCW(p: TStrings);
var c: char;
  queue: TNodeQueue;
  v, r,state,s: TNode;
  i: Integer;
  prev: tStrings;
begin
  // Trie aufbauen
  prev := tStringlist.Create;
  prev.Capacity := P.Count;

  // minimale Stringlnge bestimmen
  flmin := high(Integer);
  for i := 0 to p.Count - 1 do
  begin
    if length(p[i]) < flmin then flmin := length(p[i]);

    prev.Add(ReverseString(p[i]));
  end;

   Build(prev);

  root.SuffixLink := NIL;
  root.fLength := 0;
  root.fGS1 := 1;
  root.fGS2 := lmin;

  // Queue initialisieren
  queue := TNodeQueue.Create;
  for c := Low(Char) to High(Char) do
    if root.GetTarget(c) <> Nil then
    begin
        //Kante existiert schon - in die Queue einfgen
        v := root.GetTarget(c);
        v.fLength := 1;  // Tiefe des Knotens ist 1
        v.fGS1 := lmin;
        v.fGS2 := lmin;
        v.SuffixLink := root;
        queue.Insert(v);
    end;

  // Durchlauf durch den Trie
  While queue.First <> NIL do
  begin
      r := queue.delete;
      for c := Low(Char) to High(Char) do
      begin
          s := r.GetTarget(c);
          if s <> Nil then
          begin
              queue.Insert(s);
              s.fLength := r.fLength + 1;
              s.fGS1 := lmin;
              s.fGS2 := lmin;
              state := r.SuffixLink;
              While (state <> NIL) and (state.GetTarget(c) = NIL)  do
                state := state.SuffixLink;

              // Hier Fallunterscheidung:
              if state = NIL then
              begin
                v := root;
                s.SuffixLink := v;
              end
              else begin
                v := state.GetTarget(c);
                s.SuffixLink := v;
              end;

              // GS1 auf jeden Fall neu setzen
              v.fGS1 := min(v.fGS1, s.fLength - v.fLength);

              if s.Terminal then
              begin
                  // GS2  ndern
                  v.fGS2 := min(v.fGS2, s.fLength - v.fLength);
                  v := v.SuffixLink;
                  while (v <> NIl) do
                  begin
                      v.fGS2 := min(v.fGS2, s.fLength - v.fLength);
                      v := v.SuffixLink;
                  end;
              end;
          end;
      end;
  end;

  // Zweiten Durchlauf durch den Baum starten. Dabei fGS2 versickern lassen
  // und GS1 auf das Minimum setzen.
  for c := Low(Char) to High(Char) do
    if root.GetTarget(c) <> Nil then
    begin
        //Kante existiert schon - in die Queue einfgen
        v := root.GetTarget(c);
        v.fGS2 := min(v.fGS2, root.fGS2);
        // GS1 wird der eigentliche Shift
        if v.fGS1 > v.fGS2 then v.fGS1 := v.fGS2;
        queue.Insert(v);
    end;

  While queue.First <> NIL do
  begin
      r := queue.delete;
      for c := Low(Char) to High(Char) do
      begin
          s := r.GetTarget(c);
          if s <> Nil then
          begin
              s.fGS2 := min(s.fGS2, r.fGS2);
              if s.fGS1 > s.fGS2 then s.fGS1 := s.fGS2;
              queue.Insert(s);
          end;
      end;
  end;
  queue.Free;
end;

function TCommentzWalterTrie.Output: String;
var v: Integer;
    e: Char;
    aus: Integer;
    u: TNode;
begin
  result := '';
  // fInit ausgeben
  for e := Low(Char) to High(Char) do
    begin
        if (fRoot.fEdges[e] <> Nil) and (fRoot.GetTarget(e)<>froot) then
          result := result + Format('rootKante von %d nach %d mit Markierung %s', [
          fRoot.Idx,
          fRoot.GetTarget(e).Idx,e]) + #13#10
    end;
  result := result + Format('Ende Knoten %d, (%s)-------------',
                     [fRoot.Idx, BoolToStr(fRoot.Terminal, True)])
                   + #13#10;

  for v := 0 to fnodes.Count - 1 do
  begin
    for e := Low(Char) to High(Char) do
    begin
        if TNode(fNodes[v]).fEdges[e] <> Nil then
          result := result +  Format('Kante von %d nach %d mit Markierung %s', [
          TNode(fNodes[v]).Idx,
          TNode(fNodes[v]).GetTarget(e).Idx,e]) + #13#10
    end;

    result := result + Format('Ende Knoten %d, (Suffixlink nach %d)-------------',
                      [TNode(fNodes[v]).Idx, TNode(fNodes[v]).SuffixLink.Idx])
                     + #13#10;

    //result := result + Format('Ende Knoten %d, (%s)-------------',
    //                  [TNode(fNodes[v]).Idx, BoolToStr(TNode(fNodes[v]).Terminal, True)])
    //                 + #13#10;
    if TNode(fNodes[v]).Terminal then
    begin
        for aus := 0 to TNode(fNodes[v]).OutputList.Count - 1 do
          result := result + 'Ausgabe: ' + Inttostr(Integer(TNode(fNodes[v]).OutputList[aus])) + ', ';
        result := result + #13#10
    end;
  end;
  result := result + '============================' + #13#10;
  result := result + Format(
                     'Knoten %d: Tiefe %d, GS1 %d, GS2 %d'
                    ,[root.Idx, root.Len, root.fGS1, root.fGS2 ]) + #13#10;
  for v := 0 to fnodes.Count - 1 do
  begin
    u := TNode(fNodes[v]);
    result := result + Format(
                     'Knoten %d: Tiefe %d, GS1 %d, GS2 %d'
                    ,[u.Idx, u.Len, u.fGS1, u.fGS2 ]) + #13#10;
  end;

end;


//---------------------------------------------------------
// Klasse TMultiOracle
//---------------------------------------------------------

procedure TMultiOracle.BuildOracle(p: TStrings);
var c: char;
  queue: TNodeQueue;
  v, r,state,s: TNode;
begin
  // Trie aufbauen
  Build(p);

  queue := TNodeQueue.Create;

  root.SuffixLink := NIL;
  for c := Low(Char) to High(Char) do
    if root.GetTarget(c) <> Nil then
    begin
        //Kante existiert schon - in die Queue einfgen
        v := root.GetTarget(c);
        v.SuffixLink := root;
        queue.Insert(v);
    end;

  // Durchlauf
  While queue.First <> NIL do
  begin
      r := queue.delete;//TNode(queue.Pop);
      for c := Low(Char) to High(Char) do
      begin
          s := r.GetTarget(c);
          if s <> Nil then
          begin
              queue.Insert(s);
              state := r.SuffixLink;
              While (state <> NIL) and (state.GetTarget(c) = NIL) do
              begin
                state.SetTarget(c,s);
                state := state.SuffixLink;
              end;
              if state = NIL then
                s.SuffixLink := root
              else
              begin
                v := state.GetTarget(c); // v <> NIL !
                s.SuffixLink := v;
              end;        
          end;
      end;
  end;
  queue.Free;
end;

function TMultiOracle.Output: String;
var v: Integer;
    e: Char;
    aus: Integer;
begin
  result := '';
  // fInit ausgeben
  for e := Low(Char) to High(Char) do
    begin
        if (fRoot.fEdges[e] <> Nil) and (fRoot.GetTarget(e)<>froot) then
          result := result + Format('rootKante von %d nach %d mit Markierung %s', [
          fRoot.Idx,
          fRoot.GetTarget(e).Idx,e]) + #13#10
    end;
  result := result + Format('Ende Knoten %d, (%s)-------------',
                     [fRoot.Idx, BoolToStr(fRoot.Terminal, True)])
                   + #13#10;

  for v := 0 to fnodes.Count - 1 do
  begin
    for e := Low(Char) to High(Char) do
    begin
        if TNode(fNodes[v]).fEdges[e] <> Nil then
          result := result +  Format('Kante von %d nach %d mit Markierung %s', [
          TNode(fNodes[v]).Idx,
          TNode(fNodes[v]).GetTarget(e).Idx,e]) + #13#10
    end;
    result := result + Format('Ende Knoten %d, (Suffixlink nach %d)-------------',
                      [TNode(fNodes[v]).Idx, TNode(fNodes[v]).SuffixLink.Idx])
                     + #13#10;
    if TNode(fNodes[v]).Terminal then
    begin
        result := result + 'Ausgabe: ';
        for aus := 0 to TNode(fNodes[v]).OutputList.Count - 1 do
          result := result + Inttostr(Integer(TNode(fNodes[v]).OutputList[aus])) + ', ';
        result := result + #13#10;
    end;
  end;
end;

end.
